"use client";

import { useState, useEffect } from "react";
import { useRouter } from "next/navigation";
import { useToast } from "@/components/ui/use-toast";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { 
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { ArrowLeft, Loader2 } from "lucide-react";
import Link from "next/link";
import { MarkdownEditor } from "@/components/markdown-editor";
import { FileUploader } from "@/components/file-uploader";
import React from "react";

interface Category {
  id: string;
  name: string;
}

interface ToolData {
  id: string;
  name: string;
  slug: string;
  description: string;
  website_url: string;
  category_id: string;
  logo_url: string;
  image_url: string;
  full_description: string;
  type: string;
  category?: {
    id: string;
    name: string;
  };
}

interface EditToolClientProps {
  toolId: string;
}

// 将客户端组件分离
function EditToolClient({ toolId }: EditToolClientProps) {
  const router = useRouter();
  const { toast } = useToast();
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [isLoading, setIsLoading] = useState(true);
  const [categories, setCategories] = useState<Category[]>([]);
  const [isLoadingCategories, setIsLoadingCategories] = useState(true);
  
  const [formData, setFormData] = useState<ToolData>({
    id: toolId,
    name: "",
    slug: "",
    description: "",
    website_url: "",
    category_id: "",
    logo_url: "",
    image_url: "",
    full_description: "",
    type: "工具",
  });
  
  // 获取工具详情
  useEffect(() => {
    const fetchToolDetails = async () => {
      try {
        setIsLoading(true);
        const response = await fetch(`/api/admin/tools/${toolId}`);
        
        if (!response.ok) {
          throw new Error("获取工具详情失败");
        }
        
        const { tool } = await response.json();
        
        // 如果已有category对象，确保设置category_id
        if (tool.category && tool.category.id) {
          tool.category_id = tool.category.id;
        }
        
        setFormData({
          ...tool,
          // 确保所有必填字段都有值
          full_description: tool.full_description || "",
          logo_url: tool.logo_url || "",
          image_url: tool.image_url || "",
          type: tool.type || "工具"
        });
      } catch (error) {
        console.error("加载工具详情失败:", error);
        toast({
          title: "错误",
          description: "无法加载工具详情",
          variant: "destructive",
        });
        router.push("/admin/tools");
      } finally {
        setIsLoading(false);
      }
    };
    
    fetchToolDetails();
  }, [toolId, router, toast]);
  
  // 获取分类列表
  useEffect(() => {
    const fetchCategories = async () => {
      try {
        const response = await fetch("/api/admin/categories");
        if (!response.ok) {
          throw new Error("获取分类列表失败");
        }
        const data = await response.json();
        setCategories(data.categories || []);
      } catch (error) {
        console.error("加载分类失败:", error);
        toast({
          title: "错误",
          description: "无法加载分类列表",
          variant: "destructive",
        });
      } finally {
        setIsLoadingCategories(false);
      }
    };
    
    fetchCategories();
  }, [toast]);
  
  // 处理表单字段变化
  const handleChange = (e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement>) => {
    const { name, value } = e.target;
    setFormData((prev) => ({ ...prev, [name]: value }));
  };
  
  // 处理选择器变化
  const handleSelectChange = (name: string, value: string) => {
    setFormData((prev) => ({ ...prev, [name]: value }));
  };
  
  // 处理Markdown编辑器变化
  const handleMarkdownChange = (value: string) => {
    setFormData((prev) => ({ ...prev, full_description: value }));
  };
  
  // 处理Logo上传完成
  const handleLogoUploaded = (url: string) => {
    setFormData((prev) => ({ ...prev, logo_url: url }));
  };
  
  // 处理预览图上传完成
  const handleImageUploaded = (url: string) => {
    setFormData((prev) => ({ ...prev, image_url: url }));
  };
  
  // 处理表单提交
  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    
    // 验证必填字段
    if (!formData.name || !formData.slug || !formData.description || 
        !formData.website_url || !formData.category_id) {
      toast({
        title: "错误",
        description: "请填写所有必填字段",
        variant: "destructive",
      });
      return;
    }
    
    setIsSubmitting(true);
    
    try {
      // 提交前移除category对象，防止数据库错误
      const submitData = { ...formData };
      if ('category' in submitData) {
        delete submitData.category;
      }
      
      const response = await fetch(`/api/admin/tools/${toolId}`, {
        method: "PUT",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify(submitData),
      });
      
      if (!response.ok) {
        const errorData = await response.json();
        throw new Error(errorData.error || "更新工具失败");
      }
      
      toast({
        title: "成功",
        description: "AI工具更新成功",
      });
      
      // 重定向到工具列表页
      router.push("/admin/tools");
      
    } catch (error) {
      console.error("更新工具失败:", error);
      toast({
        title: "错误",
        description: error instanceof Error ? error.message : "更新工具失败",
        variant: "destructive",
      });
    } finally {
      setIsSubmitting(false);
    }
  };
  
  if (isLoading) {
    return (
      <div className="flex items-center justify-center h-60">
        <div className="flex flex-col items-center gap-2">
          <Loader2 className="h-8 w-8 animate-spin text-primary" />
          <p className="text-muted-foreground">加载工具信息...</p>
        </div>
      </div>
    );
  }
  
  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <div className="flex items-center gap-2">
          <Link 
            href="/admin/tools" 
            className="flex items-center text-sm text-muted-foreground hover:text-foreground"
          >
            <ArrowLeft className="h-4 w-4 mr-1" />
            返回工具列表
          </Link>
          <h1 className="text-2xl font-bold">编辑 AI 工具</h1>
        </div>
      </div>
      
      <form onSubmit={handleSubmit} className="space-y-8">
        <div className="grid grid-cols-1 gap-6 md:grid-cols-2">
          {/* 工具名称 */}
          <div className="space-y-2">
            <Label htmlFor="name">工具名称 <span className="text-destructive">*</span></Label>
            <Input
              id="name"
              name="name"
              placeholder="例如: ChatGPT"
              value={formData.name}
              onChange={handleChange}
              required
            />
          </div>
          
          {/* URL短识 */}
          <div className="space-y-2">
            <Label htmlFor="slug">URL短识 <span className="text-destructive">*</span></Label>
            <Input
              id="slug"
              name="slug"
              placeholder="例如: chatgpt"
              value={formData.slug}
              onChange={handleChange}
              required
            />
            <p className="text-xs text-muted-foreground">
              用于生成该工具的URL路径，仅允许小写字母、数字和连字符
            </p>
          </div>
          
          {/* 网站地址 */}
          <div className="space-y-2">
            <Label htmlFor="website_url">网站地址 <span className="text-destructive">*</span></Label>
            <Input
              id="website_url"
              name="website_url"
              placeholder="https://example.com"
              value={formData.website_url}
              onChange={handleChange}
              required
            />
          </div>
          
          {/* 分类 */}
          <div className="space-y-2">
            <Label htmlFor="category_id">分类 <span className="text-destructive">*</span></Label>
            <Select 
              disabled={isLoadingCategories}
              value={formData.category_id} 
              onValueChange={(value) => handleSelectChange("category_id", value)}
            >
              <SelectTrigger id="category_id">
                <SelectValue placeholder="选择分类" />
              </SelectTrigger>
              <SelectContent>
                {isLoadingCategories ? (
                  <SelectItem value="loading" disabled>
                    加载中...
                  </SelectItem>
                ) : (
                  categories.map((category) => (
                    <SelectItem key={category.id} value={category.id}>
                      {category.name}
                    </SelectItem>
                  ))
                )}
              </SelectContent>
            </Select>
          </div>
          
          {/* 工具类型 */}
          <div className="space-y-2">
            <Label htmlFor="type">工具类型</Label>
            <Select 
              value={formData.type} 
              onValueChange={(value) => handleSelectChange("type", value)}
            >
              <SelectTrigger id="type">
                <SelectValue placeholder="选择类型" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="工具">工具</SelectItem>
                <SelectItem value="网站">网站</SelectItem>
                <SelectItem value="应用">应用</SelectItem>
                <SelectItem value="API">API</SelectItem>
                <SelectItem value="其他">其他</SelectItem>
              </SelectContent>
            </Select>
          </div>
        </div>
        
        {/* 简短描述 */}
        <div className="space-y-2">
          <Label htmlFor="description">简短描述 <span className="text-destructive">*</span></Label>
          <Textarea
            id="description"
            name="description"
            placeholder="简短描述该工具的主要功能和特点（最多500字符）"
            value={formData.description}
            onChange={handleChange}
            rows={4}
            required
            maxLength={500}
          />
        </div>
        
        {/* 工具Logo上传 */}
        <div className="space-y-2">
          <Label>工具Logo</Label>
          <FileUploader 
            bucketName="tool-logos"
            accept="image/jpeg,image/png,image/svg+xml,image/webp"
            maxSizeMB={2}
            value={formData.logo_url}
            onUploadComplete={handleLogoUploaded}
          />
          <p className="text-xs text-muted-foreground">
            上传工具的Logo图片，最大2MB，支持JPG、PNG、SVG和WebP格式
          </p>
        </div>
        
        {/* 工具预览图上传 */}
        <div className="space-y-2">
          <Label>工具预览图</Label>
          <FileUploader 
            bucketName="tool-previews"
            accept="image/jpeg,image/png,image/webp"
            maxSizeMB={5}
            value={formData.image_url}
            onUploadComplete={handleImageUploaded}
          />
          <p className="text-xs text-muted-foreground">
            上传工具的预览图或截图，最大5MB，支持JPG、PNG和WebP格式
          </p>
        </div>
        
        {/* 详细介绍 Markdown编辑器 */}
        <div className="space-y-2">
          <Label htmlFor="full_description">详细介绍（可选）</Label>
          <MarkdownEditor
            value={formData.full_description}
            onChange={handleMarkdownChange}
          />
          <p className="text-xs text-muted-foreground">
            支持Markdown格式，可以使用标题、列表、链接等丰富的格式
          </p>
        </div>
        
        {/* 提交按钮 */}
        <div className="flex justify-end">
          <Button type="submit" disabled={isSubmitting}>
            {isSubmitting ? (
              <>
                <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                保存中...
              </>
            ) : "保存修改"}
          </Button>
        </div>
      </form>
    </div>
  );
}

// 页面组件，解析 params 并传给客户端组件
export default function EditToolPage({ params }: { params: { id: string } | Promise<{ id: string }> }) {
  // 使用 React.use() 解包 params
  const resolvedParams = params instanceof Promise ? React.use(params) : params;
  return <EditToolClient toolId={resolvedParams.id} />;
} 